<?php
class BackendApi
{
  private $options;
  private $backendUrl;
  private $pluginVersion;

  public function __construct($options, $backendUrl, $pluginVersion)
  {
    $this->options = $options;
    $this->backendUrl = $backendUrl;
    $this->pluginVersion = $pluginVersion;
  }

  public function bootstrap($ownerEmail)
  {
    $domain = get_option('siteurl');
    $curl = curl_init();
    $apiBackendUrl = $this->backendUrl . '/api/keys/bootstrap';
    
    $data = [
      'domain' => $domain,
      'ownerEmail' => $ownerEmail
    ];
    $jsonData = json_encode($data);

    $httpHeaders = array(
      "Domain: " . $domain,
      "Content-Type: application/json",
      "Content-Length: " . strlen($jsonData)
    );

    curl_setopt_array($curl, array(
      CURLOPT_URL => $apiBackendUrl,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_CUSTOMREQUEST => "POST",
      CURLOPT_HTTPHEADER => $httpHeaders,
      CURLOPT_POSTFIELDS => $jsonData,
      CURLOPT_TIMEOUT => 10
    ));

    $response = curl_exec($curl);
    $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);

    if ($http_code === 200) {
      $data = json_decode($response);
      if (isset($data->apiKey)) {
        $this->options['api_key'] = $data->apiKey;
        $this->options['owner_email'] = $ownerEmail;
        update_option('dbi_bemoto_plugin_options', $this->options);
        return $data->apiKey;
      }
    }
    return null;
  }

  public function getData($locale, $shortCodeParams = "", $endpoint = "/api/cars")
  {
    $phoneNumber  = $this->options['phone_number'];
    $username = $this->options['username'] ?? "";
    $pass = $this->options['password'] ?? "";
    $customerId  = $this->options['customer_id'] ?? "";
    $dataCredentials = $this->options['dataCredentials'];
    if (!isset($dataCredentials)) {
      $dataCredentials[] = [
        'clientNumber' => $customerId,
        'userName' => $username,
        'password' => $pass,
      ];
    }
    $mobileDeAuthList = array_map(function ($credential) {
      return (object)[
        'type' => $credential['type'],
        'clientNumber' => $credential['clientNumber'],
        'userName' => $credential['userName'],
        'mobileDeAuth' => base64_encode($credential['userName'] . ':' . $credential['password']),
      ];
    }, $dataCredentials);
    $domain = get_option('siteurl');

    $apiKey = $this->options['api_key'] ?? "";

    $isDemoMode = empty($apiKey);

    $curl = curl_init();
    $apiBackendUrl = $this->backendUrl . $endpoint;
    $data = [
      'DataProvidersCredentials' => $mobileDeAuthList,
      'PhoneNumber' => $phoneNumber
    ];
    if (!empty($shortCodeParams)) {
      $data['ShortCodeParams'] = base64_encode($shortCodeParams);
    }

    $jsonData = json_encode($data);
    // as of now BemotoPluginParams is mainly used for passing proper referer
    $BemotoPluginParams = "";
    if (isset($_POST['bemoto-plugin-params'])) {
      $BemotoPluginParams = $_POST['bemoto-plugin-params'];
    }
    $httpHeaders = array(
      "Domain: " . $domain,
      "PluginVersion: " . $this->pluginVersion,
      "BemotoPluginParams: " . $BemotoPluginParams,
      "Content-Type: application/json",
      "Accept-Language: " . $locale,
      "Content-Length: " . strlen($jsonData)
    );
    if (!$isDemoMode) {
      $httpHeaders[] = "X-Bemoto-API-Key: " . $apiKey;
    }

    curl_setopt_array($curl, array(
      CURLOPT_URL => $apiBackendUrl,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_ENCODING => "",
      CURLOPT_MAXREDIRS => 10,
      CURLOPT_FOLLOWLOCATION => true,
      CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
      CURLOPT_CUSTOMREQUEST => "POST",
      CURLOPT_HTTPHEADER => $httpHeaders,
      CURLOPT_CONNECTTIMEOUT => 5,
      CURLOPT_TIMEOUT => 2 * 60,
      CURLOPT_POSTFIELDS => $jsonData,
    ));
    $result = curl_exec($curl);
    $curl_errno = curl_errno($curl);
    $curl_error = curl_error($curl);
    $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);

    $obj = new stdClass;
    $obj->status = $http_code;
    $obj->content = $result;
    $obj->errNo = $curl_errno;
    $obj->error = $curl_error;

    return $obj;
  }

  public function sendDebugLog($message)
  {
    // The Firebase Cloud Function endpoint for logs
    // Assumes $this->backendUrl is the base URL for the API, e.g., "https://app.bemoto.eu/api"
    $logEndpoint = $this->backendUrl . '/api/bemoto-logs';

    $domain = get_option('siteurl');
    $data = [
      'timestamp' => current_time('mysql'), // WordPress function for current time
      'message'   => $message,
    ];

    $jsonData = json_encode($data);

    $curl = curl_init();

    $httpHeaders = array(
      "Domain: " . $domain,
      "PluginVersion: " . $this->pluginVersion,
      "Content-Type: application/json",
      "Content-Length: " . strlen($jsonData)
    );

    curl_setopt_array($curl, array(
      CURLOPT_URL => $logEndpoint,
      CURLOPT_RETURNTRANSFER => true,
      CURLOPT_ENCODING => "",
      CURLOPT_MAXREDIRS => 10,
      CURLOPT_TIMEOUT => 5, // Short timeout for non-critical logging
      CURLOPT_FOLLOWLOCATION => true,
      CURLOPT_HTTP_VERSION => CURL_HTTP_VERSION_1_1,
      CURLOPT_CUSTOMREQUEST => "POST",
      CURLOPT_POSTFIELDS => $jsonData,
      CURLOPT_HTTPHEADER => $httpHeaders,
      CURLOPT_FRESH_CONNECT => true, // Don't reuse connections
      CURLOPT_FORBID_REUSE => true, // Don't reuse connections
    ));

    $result = curl_exec($curl);
    $curl_errno = curl_errno($curl);
    $curl_error = curl_error($curl);
    $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
    curl_close($curl);

    if ($curl_errno !== 0 || $http_code !== 200) {
      // Log errors to the standard WordPress error log if sending to Firebase fails
      error_log('Bemoto Debug: Failed to send log to Firebase. Error: ' . $curl_error . ' (Code: ' . $curl_errno . '), HTTP Code: ' . $http_code . ', Response: ' . $result);
    }
  }
}
