<?php

/**
 * Plugin Name: bemoto.eu
 * Plugin URI: https://bemoto.eu
 * Description: Wordpress + Mobile.de integration plugin
 * Version: 1.0.0
 * Author: bemoto.eu | Vanillia Marcin Klimczak
 * Author URI: https://bemoto.eu
**/

$configs = include_once 'config.php';
require_once 'backendApi.php';
require_once 'shortcodes.php';
require_once 'pages.php';
require_once 'utils.php';
require_once 'dataSync.php';
require_once 'carDetails.php';
require_once 'cacheTester.php';

$backendUrl = $configs['function-base-url'];
$environment = strpos($backendUrl, 'app.bemoto.eu') !== false ? "production" : "debug";
$isProd = $environment == "production";
$optionsKey = 'dbi_bemoto_plugin_options';
$pageCacheKey = 'dbi_bemoto_page_cache';

$plugin_path = plugin_dir_path(__FILE__);
error_log('Bemoto Debug: Plugin path: ' . $plugin_path);
$plugin_data = get_file_data($plugin_path . "bemoto.php", array('Version' => 'Version'), false);
error_log('Bemoto Debug: Plugin data: ' . print_r($plugin_data, true));
$pluginVersion = $plugin_data['Version'];
$pluginSettingsBootstrapFile = getPathFor('index.html');

/**
 * REGISTER PLUGIN SETTINGS
 **/
function Bemoto_Plugin_Register_settings()
{
  add_options_page('Bemoto', 'Bemoto', 'manage_options', 'dbi-bemoto-plugin', 'dbi_render_plugin_settings_page');
}

function get_settings_view()
{
  global $isProd, $pluginSettingsBootstrapFile, $backendUrl;

  if (!$isProd) {
    return;
  }

  $cacheTime = 24 * 60 * 60; // 24 hours in seconds

  if (!file_exists($pluginSettingsBootstrapFile) || time() - filemtime($pluginSettingsBootstrapFile) >= $cacheTime) {
    // If the cache file doesn't exist or is older than 24 hours, download the HTML file and cache it
    try {
      $result = getUrlContent($backendUrl . '/assets/plugin-settings/index.html');
      if ($result->status != 200) {
        echo "<br/>Error when initializing plugin settings<br/> Code: " . $result->status . "<br/>Message: " . $result->content;
        echo '<h1>Could not load plugin settings page, please send a screenshot of this page to our support email <a href="mailto:support@bemoto.eu">support@bemoto.eu</a></h1>';
      } else {
        $html = $result->content;
        if (!empty($html)) {
          // Ensure the directory exists before writing the file
          $dirPath = dirname($pluginSettingsBootstrapFile);
          if (!file_exists($dirPath)) {
            wp_mkdir_p($dirPath);
          }
          file_put_contents($pluginSettingsBootstrapFile, $html);
        }
      }
    } catch (Exception $e) {
      echo "Error when initializing bemoto settings: " . $e->getMessage();
      echo '<h1>Could not load plugin settings page, please send a screenshot of this page to our support email <a href="mailto:support@bemoto.eu">support@bemoto.eu</a></h1>';
    }
  }
}

function getCachingPlugins()
{
  $all_plugins = get_plugins();
  $active_plugins = get_option('active_plugins');

  // Initialize an array to hold the data you want to collect
  $plugins_data = [];

  foreach ($active_plugins as $plugin_path) {
    // Extract the plugin's directory name as plain plugin name
    $plainPluginName = explode('/', $plugin_path)[0];

    // Assuming you want to fetch the plugin data based on the path
    if (isset($all_plugins[$plugin_path])) {
      $plugin_data = $all_plugins[$plugin_path];

      // You can add any other properties from $plugin_data or manipulate it as needed
      $plugins_data[] = [
        'Name' => $plugin_data['Name'] ?? 'N/A',
        'Version' => $plugin_data['Version'] ?? 'N/A',
        // Add any other data you're interested in
        'Directory' => $plainPluginName,
        'PluginURI' => $plugin_data['PluginURI'] ?? 'N/A',
      ];
    }
  }

  // For demonstration, let's encode this data as JSON
  return json_encode($plugins_data, JSON_PRETTY_PRINT | JSON_UNESCAPED_SLASHES);
}

function dbi_render_plugin_settings_page()
{
  global $backendUrl, $environment, $plugin_path, $pluginSettingsBootstrapFile,
  $pluginVersion, $cachingPlugins;

  $cachingPlugins = getCachingPlugins();

  require_once("initialData.php");

  get_settings_view();

  if (file_exists($pluginSettingsBootstrapFile)) {
    include_once($pluginSettingsBootstrapFile);
  }
  else {
    echo '<h1>Could not load plugin settings page, please send a screenshot of this page to our support email <a href="mailto:support@bemoto.eu">support@bemoto.eu</a></h1>';

    $active_plugins = get_option('active_plugins');
    $pluginListText = "";

    foreach ($active_plugins as $plugin) {
      $plainPluginName = explode('/', $plugin)[0];
      $pluginListText = $pluginListText . $plainPluginName . "<br/> ";
    }

    echo 'Plugin list:<br/>' . $pluginListText;
  }
}

add_action('admin_menu', 'Bemoto_Plugin_Register_settings');

function dbi_register_settings()
{
  global $optionsKey;
  register_setting($optionsKey, $optionsKey);
  add_settings_section('api_settings', '', '', 'dbi_bemoto_plugin');
}

add_action('admin_init', 'dbi_register_settings');

// Function to check version and flush rules if needed on update
function bemoto_check_version_and_flush_rules() {
    global $pluginVersion; // Make sure $pluginVersion is accessible

    $stored_version_option_name = 'bemoto_plugin_version';
    $stored_version = get_option($stored_version_option_name);

    // Check if the stored version is different from the current version or if it's not set yet
    if ($stored_version !== $pluginVersion) {
        // Version mismatch or first run after adding this check, flush rules
        // Note: dynamic_car_rewrite_rules() adds the rules, flush_rewrite_rules() saves them.
        // We might need to ensure rules are added before flushing here if this runs before init.
        // Calling the existing flush function which internally calls dynamic_car_rewrite_rules might be safer.
        flush_car_rewrite_rules(); // Use the existing function that handles adding rules too
        // Update the stored version to the current version
        update_option($stored_version_option_name, $pluginVersion);
    }
}
// Hook this check to admin_init
add_action('admin_init', 'bemoto_check_version_and_flush_rules');

function bemoto_handle_plugin_update($upgrader_object, $options) {
    // Check if the update was for a plugin
    if ($options['type'] === 'plugin') {
        // Flush rewrite rules
        flush_car_rewrite_rules();
    }
}
add_action('upgrader_process_complete', 'bemoto_handle_plugin_update', 10, 2);

function bemoto_get_current_vehicle_filepath() {
    $hash = '';
    $locale = 'en'; // Default locale
    if (isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) && is_string($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
      $locale = substr($_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2);
    }
    $carId = get_query_var('carId');
    if (!is_numeric($carId)) {
        $carId = null; // Ensure $carId is null if not a valid number
    }

    if ($carId) {
        $hash = 'details/' . $carId;
    } else {
        $currentUrl = $_SERVER['REQUEST_URI'];
        $path = parse_url($currentUrl, PHP_URL_PATH);
        $path = trim($path, '/');

        if (!empty($path)) {
            $pathSegments = explode('/', $path);
            $hash = end($pathSegments); // Get the last segment
        } else {
            if (is_front_page() || is_home()) {
                $hash = "bemoto-home-page";
            }
        }
    }

    if (empty($hash)) {
        return null;
    }

    return getPathFor('vehicle.html', $hash, $locale);
}

function bemoto_list_shortcode($atts)
{
  if (!defined('DONOTCACHEPAGE')) {
    define('DONOTCACHEPAGE', true);
  }
  global $isProd, $optionsKey, $backendUrl, $pluginVersion;
  $options = get_option($optionsKey);
  $mobileDataFetcher = new BackendApi($options, $backendUrl, $pluginVersion);
  $customStyle = "";
  $customSettings = "";
  $hash = "";
  $locale = 'en'; // Default locale
  if (isset($_SERVER['HTTP_ACCEPT_LANGUAGE']) && is_string($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
    $locale = substr($_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2);
  }
  $has_attributes = !empty($atts) && is_array($atts);
  $json_attributes = json_encode($atts);

  // Initialize from shortcode attributes, then fallback to query vars
  $carId = $atts['carid'] ?? null;
  $source = $atts['source'] ?? null;
  $carName = $atts['carname'] ?? null;
  $pageId = $atts['pageid'] ?? null;

  // If not provided by attributes, try to get from query vars
  if (is_null($carId)) {
      $carId_raw = get_query_var('carId');
      if (is_numeric($carId_raw)) {
          $carId = (int)$carId_raw;
      }
  }
  if (is_null($source)) {
      $source_raw = get_query_var('source');
      if (is_string($source_raw)) {
          $source = preg_replace('/[^a-fA-F0-9]/', '', $source_raw);
      }
  }
  if (is_null($carName)) {
      $carName = get_query_var('carName');
  }
  if (is_null($pageId)) {
      $pageId_raw = get_query_var('page_id');
      if (is_numeric($pageId_raw)) {
          $pageId = (int)$pageId_raw;
      }
  }

  $shortCodeParams = $json_attributes;

  $currentUrl = $_SERVER['REQUEST_URI'] ?? ''; // Initialize $currentUrl
  $path = parse_url($currentUrl, PHP_URL_PATH);
  $path = trim($path, '/'); // Remove leading/trailing slashes

  $basePath = '/'; // Initialize $basePath
  $hash = "bemoto-home-page"; // Default for home page

  if (!empty($path)) {
    $pathSegments = explode('/', $path);
    $hash = end($pathSegments); // Get the last segment
    $hash = sanitize_file_name($hash); // Sanitize the hash
    $basePath = '/' . $path . '/'; // Reconstruct basePath
  } else {
    if (is_front_page() || is_home()) {
      $hash = "bemoto-home-page";
      $hash = sanitize_file_name($hash); // Sanitize for consistency
      $basePath = '/'; // Explicitly set basePath to root
    }
  }

  // Skip processing for admin area requests and wp-json API requests
  if (is_admin() || (!empty($path) && strpos($path, 'wp-json') === 0)) {
    $mobileDataFetcher->sendDebugLog("Skipping Bemoto shortcode processing for admin/API URL: ".$currentUrl);
    return; // Skip processing
  }

  $mobileDataFetcher->sendDebugLog("currentUrl: ".$currentUrl.", basePath: ".$basePath.", hash: ".$hash);

  if ($carId) {
    $hash = sanitize_file_name((string)$carId); // Sanitize carId before assigning to hash
    // Back anchor logic
    // Set is_single and is_page to true to make WordPress recognize this as a single page
    global $wp_query;
    $wp_query->is_single = true;
    $wp_query->is_page = true;
    $wp_query->is_404 = false;
    $wp_query->query_vars['carId'] = $carId;
    $wp_query->query_vars['source'] = $source;
    $wp_query->query_vars['carName'] = $carName; // Use the resolved carName

    error_log('Bemoto Debug: REQUEST_URI: ' . $_SERVER['REQUEST_URI']);
    error_log('Bemoto Debug: carId: ' . $carId);

    $currentUrlPath = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
    $pathSegments = array_filter(explode('/', trim($currentUrlPath, '/'))); // Split and remove empty segments
    error_log('Bemoto Debug: initial pathSegments: ' . implode(', ', $pathSegments));
    
    $backLinkUrl = '/'; // Default to root

    // Check if we are on a car details page (carId is set) and there are enough segments
    // A car details URL will have at least two segments after the base path: MAKE and SLUG-ID-SOURCE
    // e.g., /demo/FORD/fiesta-cool-ID-SOURCE/
    // or /APRILIA/sr-gt-ID-SOURCE/
    if ($carId && count($pathSegments) >= 2) {
        // Remove the last two segments (car slug and car make)
        array_pop($pathSegments); // Removes the slug-id-source
        array_pop($pathSegments); // Removes the car make
        
        // Reconstruct the path. If no segments left, it means the base was root.
        if (!empty($pathSegments)) {
            $backLinkUrl = '/' . implode('/', $pathSegments) . '/';
        } else {
            $backLinkUrl = '/'; // If pathSegments is empty, it means the base was root
        }
    }
    error_log('Bemoto Debug: final backLinkUrl: ' . $backLinkUrl);

    $backAnchor = '<div style="display: inline-block;padding: 10px 30px; border-radius: 10px; background-color: #ddd; margin-bottom: 50px; font-size: 30px;"><a href="' . esc_url($backLinkUrl) . '">zurück</a></div>';
    // bemoto/details/123/pl/vehicle.html
    $currentVehiclesFilePathHtml = bemoto_get_current_vehicle_filepath();
    get_vehicle_details($locale, $carId, $shortCodeParams, $source, false);

    $ajax_url = admin_url('admin-ajax.php?action=bemoto_refresh&carId=' . $carId.'&hash='.$hash.'&source='.esc_attr($source));
    $customJsCode = "<script type='text/javascript'>window.ajaxurl='" . $ajax_url . "';</script>";

    $isFileExisting = file_exists($currentVehiclesFilePathHtml);
    $content = "no data, check plugin settings";
    if ($isFileExisting) {
      $content = file_get_contents($currentVehiclesFilePathHtml);
    }

    if (isset($options['custom_settings_json'])) {
      $customSettingsJson = $options['custom_settings_json'];
      $customSettings = json_decode($customSettingsJson);

      if (is_object($customSettings)) {
        $customStyle = "<style>" .
          ".bemoto-plugin { --radius: ".$customSettings->borderRadius ."px}".
          "</style>";
      }
    }

    return
      '<div class="bemoto-plugin override" style="max-width: 100%;" data-nonce="' . wp_create_nonce('bemoto_dynamic_content') . '">'
        . $backAnchor . $customJsCode . $customStyle . $content .
        '</div>';
  }


  $currentVehiclesFilePathJson = getPathFor('vehicle.json', $hash);
  $currentVehiclesFilePathHtml = bemoto_get_current_vehicle_filepath();

  if (empty($currentVehiclesFilePathHtml)) {
    $currentVehiclesFilePathHtml = getPathFor('vehicle.html', 'demo');
  }
  $mobileDataFetcher->sendDebugLog("$currentVehiclesFilePathJson: ". $currentVehiclesFilePathJson);
  if (!empty($hash)) {
    /*echo '<br/>Using hash: '.$hash; */
    // Save the JSON attributes in a file
    // Ensure the directory exists before writing the file
    $dirPath = dirname($currentVehiclesFilePathJson);
    if (!file_exists($dirPath)) {
      wp_mkdir_p($dirPath);
    }
    $result = @file_put_contents($currentVehiclesFilePathJson, $json_attributes);

    if ($result === false) {
      // Get the last error
      $error = error_get_last();
      log_error('Error writing to file: ' . $error['message']);
    }
  }

  $ajax_url = admin_url('admin-ajax.php?action=bemoto_refresh&hash=' . $hash);
  $customJsCode = "<script type='text/javascript'>window.ajaxurl='" . $ajax_url . "';</script>";
  if (isset($options['custom_settings_json'])) {
    $customSettingsJson = $options['custom_settings_json'];
    $customSettings = json_decode($customSettingsJson);

    if (is_object($customSettings)) {
      $customStyle = "<style>" .
        ".bemoto-plugin.override { background-color: " . $customSettings->pageBgColor . ";}" .
        ".bemoto-plugin.override .ad { box-shadow: 7px 8px 21px 0px " . $customSettings->shadowColor . ";}" .
        ".bemoto-plugin.override .ad:hover { scale: 1.05; box-shadow: " . $customSettings->hoverBoxShadow . " !important;}" .

        ".bemoto-plugin.override .ad { background-color: " . $customSettings->bgColor . ";color:" . $customSettings->textColor . ";border-radius: " . $customSettings->borderRadius . "px}" .
        ".bemoto-plugin.override .ad .title { color:" . $customSettings->titleTextColor . ";}" .
        ".bemoto-plugin.override .ad .subtitle { color:" . $customSettings->subTitleTextColor . ";}" .
        ".bemoto-plugin.override .ad .price { color:" . $customSettings->priceColor . ";}" .

        ".callMe.override .callToAction { background-color: " . $customSettings->bgIconColor . "; }" .
        ".callMe.override .callToAction svg { fill:" . $customSettings->iconColor . ";}" .
        ".callMe.override #phoneNoSlider { background-color:" . $customSettings->phoneNoBackgroundColor . ";color:" . $customSettings->phoneNoTextColor . ";border-radius: " . $customSettings->phoneNoBorderRadius . "px; }" .
        "</style>";
    }
  }

  if ($isProd) {
    $content = getVehicleFileContent($customJsCode . $customStyle, $customSettings, $hash, $locale);

    if (isset($content)) {
      return $content;
    }
  }

  get_vehicle_data($locale, false, $hash);
  $isFileExisting = file_exists($currentVehiclesFilePathHtml);
  $content = "no data, check plugin settings";
  if ($isFileExisting) {
    $content = file_get_contents($currentVehiclesFilePathHtml);
    if (is_object($customSettings) && $customSettings->shouldMakeSlim) {
      $content = str_replace('div class="ads"', 'div class="ads make-slim"', $content);
    }
  }

  $nonce = wp_create_nonce('bemoto_dynamic_content');
  $return_string = $customJsCode . $customStyle . $content;
  $return_string = str_replace('<div class="bemoto-plugin', '<div class="bemoto-plugin override" data-nonce="' . $nonce . '"', $return_string);

  return $return_string;
}


// get local disk file or refresh it if wasn't refreshed for 24 hours
function getVehicleFileContent($customJsAndStyle, $customSettings, $hash, $locale)
{
  $currentVehiclesFilePath = getPathFor('vehicle.html', $hash, $locale);
  $isFileExisting = file_exists($currentVehiclesFilePath);
  if ($isFileExisting) {
    $fileLastChangeDate = new DateTime();
    $fileLastChangeDate->setTimestamp(filemtime($currentVehiclesFilePath));

    $now = new DateTime();
    $now->setTimestamp(time());

    $secondsElapsed = $now->getTimestamp() - $fileLastChangeDate->getTimestamp();
    $minutes = 60;

    $secondsTriggeringRefresh = 24 * $minutes * 60;

    if ($secondsElapsed >= $secondsTriggeringRefresh) {
      get_vehicle_data($locale, false, $hash);
    }
  } else {
    get_vehicle_data($locale, false, $hash);
  }

  $content = file_get_contents($currentVehiclesFilePath);
  if (is_object($customSettings) && $customSettings->shouldMakeSlim) {
    $content = str_replace('div class="ads"', 'div class="ads make-slim"', $content);
  }
  return $customJsAndStyle . $content;
}

function log_info_in_browser($message)
{
  // echo '<script>console.info("Bemoto Wordpress Plugin: ' . $message . '");</script>';
}
function log_error_in_browser($message)
{
  // echo '<script>console.error("Bemoto Wordpress Plugin: ' . $message . '");</script>';
}

function log_error($message)
{
  $errorFilePath = getPathFor('errors.html');
  $log_line = date_format(new DateTime(), "d.m.Y H:i:s") . " " . $message . "</br>";
  $isFileExisting = file_exists($errorFilePath);

  if ($isFileExisting) {
    $logTruncateSizeInKB = 10;
    $logSizeLimit = $logTruncateSizeInKB * 1024;
    $logContent = file_get_contents($errorFilePath);
    if (strlen($logContent) > $logSizeLimit) {
      $logContent = substr($logContent, 0, $logSizeLimit);
    }

    $log_line .= $logContent;
  }
  // Ensure the directory exists before writing the file
  $dirPath = dirname($errorFilePath);
  if (!file_exists($dirPath)) {
    wp_mkdir_p($dirPath);
  }
  file_put_contents($errorFilePath, $log_line);
}

function read_with_limit($filename, $sizeInKB = 256)
{
  $result = "";
  $maxFileSize = $sizeInKB * 1024; // 256 KB in bytes

  if (file_exists($filename)) {
    $fileSize = filesize($filename);

    // Open the file and read up to the maximum size allowed
    if (file_exists($filename)) {
      $fileSize = filesize($filename);

      // Open the file and read up to the maximum size allowed
      if ($fileSize > 0) {
        // Open the file in read mode
        $handle = fopen($filename, "r");
        if ($handle) {
          // Move the file pointer to the start of the last 256 KB or to the beginning if the file is smaller
          if ($fileSize > $maxFileSize) {
            fseek($handle, -$maxFileSize, SEEK_END);
          } else {
            // If the file is smaller than 256 KB, read from the beginning
            fseek($handle, 0, SEEK_SET);
          }

          // Read up to 256 KB from the file
          $result = fread($handle, $maxFileSize);
          fclose($handle);
        }
      }
    }
  }
  return $result;
}

function get_vehicle_data($locale, $forceRefresh = false, $hash = "")
{
  global $optionsKey, $backendUrl, $pluginVersion;

  $options = get_option($optionsKey);
  $mobileDataFetcher = new BackendApi($options, $backendUrl, $pluginVersion);
  $shortCodeParams = "";
  error_log("refresh data for hash ".$hash);
  $currentVehiclesFilePathJson = getPathFor('vehicle.json', $hash);
  $currentVehiclesFilePathHtml = getPathFor('vehicle.html', $hash, $locale);
  if (empty($currentVehiclesFilePathHtml)) {
    $currentVehiclesFilePathHtml = getPathFor('vehicle.html', 'demo');
  }
  if (!empty($hash)) {
    $shortCodeParams = file_get_contents($currentVehiclesFilePathJson);
  }

  $result = $mobileDataFetcher->getData($locale, $shortCodeParams);
  log_info_in_browser('got results from bemoto server');

  if ($result->errNo === CURLE_OPERATION_TIMEOUTED || $result->errNo === CURLE_COULDNT_CONNECT) {
    // try once more after first timeout
    $result = $mobileDataFetcher->getData($locale, $shortCodeParams);

    // if timed out second time then we don't try anymore
    if ($result->errNo === CURLE_OPERATION_TIMEOUTED || $result->errNo === CURLE_COULDNT_CONNECT) {
      log_error("Couldn't connect to server, timeout happened. Error code: " . $result->errNo . ", message: " . $result->error);
      return;
    }
  }

  //process result if there was no timeouts
  switch ($result->status) {
    case 401:
      log_error("Unauthorized request to backend. Please check your settings.");
      break;
    case 200:
      // Ensure the directory exists before writing the file
      $dirPath = dirname($currentVehiclesFilePathHtml);
      if (!file_exists($dirPath)) {
        wp_mkdir_p($dirPath);
      }
      $fileSaveResult = @file_put_contents($currentVehiclesFilePathHtml, $result->content);
      if ($fileSaveResult === false) {
        $error = error_get_last();
        $errorMessage =  $error['message'];
        log_error_in_browser($errorMessage);
        log_error('Error writing to file: ' . $error['message']);
      }
      break;
    default:
      $errorMessage = $result->content;
      log_error_in_browser($errorMessage);
      if ($result->status == 0) {
        $errorMessage = "Can't connect to firebase";
      } else {
        log_error("Error status: " . $result->status);
      }
      log_error($errorMessage);
  }
}

add_shortcode('bemoto_plugin', 'bemoto_list_shortcode');

add_action('init', function() use ($pluginVersion) {
    activate_au($pluginVersion);
});
function activate_au($pluginVersionArgument = null)
{
  global $pluginVersion;
  if ($pluginVersionArgument) {
    $pluginVersion = $pluginVersionArgument;
  }
  require_once('autoupdate.php');

  new mishaUpdateChecker($pluginVersion);
}

function bemoto_refresh()
{
  global $isProd;
  $minutes = 60;

  // The $_REQUEST contains all the data sent via AJAX from the Javascript call
  // if (isset($_REQUEST)) {
  if (isset($_REQUEST) && isset($_REQUEST['hash'])) {
    $hash = $_GET['hash'] ?? '';
    // Sanitize the hash to prevent path traversal and arbitrary file creation
    $hash = sanitize_file_name($hash);
    $what = isset($_GET['what']) ? $_GET['what'] : null;
    $locale = 'en'; // Default locale
    if (isset($_SERVER['HTTP_ACCEPT_LANGUAGE'])) {
      $locale = substr($_SERVER['HTTP_ACCEPT_LANGUAGE'], 0, 2);
    }

    if (is_front_page() || is_home()) {
      $hash = "bemoto-home-page";
    }

    // Get all registered shortcodes and their slugs
    $all_shortcodes_data = find_all_shortcodes();
    $valid_shortcode_slugs = array_map(function($item) {
        return $item['postName'];
    }, $all_shortcodes_data);

    // If the current hash is not 'bemoto-home-page' and not in the list of valid slugs,
    // then it's an invalid shortcode slug.
    if ($hash !== "bemoto-home-page" && !in_array($hash, $valid_shortcode_slugs)) {
        // Log the invalid hash and terminate the AJAX request
        log_error("Invalid shortcode slug received in AJAX refresh: ".$hash);
        wp_send_json_error(['message' => 'Invalid request hash.'], 400);
        die();
    }

    $carId = isset($_GET['carId']) ? $_GET['carId'] : null;
    $source = isset($_GET['source']) ? $_GET['source'] : null;
    if ($carId) {
      $currentVehiclesFilePathJson = getPathFor('vehicle.json', $hash);
      $currentVehiclesFilePath = getPathFor('vehicle.html', 'details/' . $carId, $locale);
      $refreshingMarkerFile = getPathFor('bemoto-is-refreshing-data.marker', 'details/' . $carId, $locale);

      if (!empty($hash)) {
        $shortCodeParams = file_get_contents($currentVehiclesFilePathJson);
      }
      syncIt($refreshingMarkerFile, $currentVehiclesFilePath, function () use ($carId, $shortCodeParams, $source, $hash, $locale) {
        $minutesBetweenSync = 24 * 60;
        get_vehicle_details($locale, $carId, $shortCodeParams, $source, false, $minutesBetweenSync);
      });
      die();
    }

    $refreshingMarkerFile = getPathFor('bemoto-is-refreshing-data.marker', $hash, $locale);
    $currentVehiclesFilePath = getPathFor('vehicle.html', $hash, $locale);
    syncIt($refreshingMarkerFile, $currentVehiclesFilePath, function () use ($hash, $locale) {
      get_vehicle_data($locale, false, $hash);
    });
  }

  // Always die in functions echoing AJAX content
  die();
}

function handle_bemoto_options()
{
  // Add capability check for security
  if (!current_user_can('manage_options')) {
    wp_die('You do not have sufficient permissions to access this page.');
  }

  global $optionsKey, $pageCacheKey;
  switch ($_SERVER['REQUEST_METHOD']) {
    case 'GET':
      $what = isset($_GET['what']) ? $_GET['what'] : null;
      // Conditional logic based on the value of 'what'
      switch ($what) {
        case 'diagnostic-data':
          $includeDebugInfo = isset($_GET['includeDebugInfo']) ? $_GET['includeDebugInfo'] : null;
          // Code to handle diagnostic data request
          $errorFilePath = getPathFor('/errors.html');

          $logContent = "";
          $isFileExisting = file_exists($errorFilePath);
          if ($isFileExisting) {
            $logContent = file_get_contents($errorFilePath);
          }

          $response = [
            "errors" => $logContent
          ];

          if ($includeDebugInfo) {
            $nativeErrorLogPath = ini_get('error_log');
            $nativeLogContent = read_with_limit($nativeErrorLogPath, 256);

            if (!empty($nativeLogContent)) {
              $response["nativeErrors"] = $nativeLogContent;
            }

            // Capture phpinfo output as a string
            ob_start();
            phpinfo();
            $phpinfoContent = ob_get_clean();
            if (!empty($phpinfoContent)) {
              $response["phpInfo"] = $phpinfoContent;
            }
          }

          echo json_encode($response);
          break;
        case 'vehicle-info':
          // Code to handle vehicle info request
          echo json_encode(["status" => "Vehicle info retrieved"]);
          break;
        case 'shortcodes':
          // Code to handle vehicle info request
          $shortcodes = find_all_shortcodes();
          $bemotoOptionsInDb = get_option($pageCacheKey, []);
          // Loop through $shortcodes and enrich each with diffInSeconds from $bemotoOptionsInDb
          foreach ($shortcodes as &$shortcode) {
            $postId = $shortcode['postId'];

            // Add diffInSeconds if the postId exists in $bemotoOptionsInDb
            if (isset($bemotoOptionsInDb[$postId])) {
              $shortcode['cacheTestResult'] = $bemotoOptionsInDb[$postId];
            } else {
              $shortcode['cacheTestResult'] = null; // Or default value if missing
            }
          }
          // Unset reference to prevent unexpected issues
          unset($shortcode);
          echo json_encode($shortcodes);
          break;
        case 'changelog':
          // Fetch changelog directly from info.json
          $info_json_url = 'https://app.bemoto.eu/info.json'; // Assuming this is the public URL for info.json
          $response = wp_remote_get($info_json_url);

          $changelog_content = '';
          if (!is_wp_error($response) && wp_remote_retrieve_response_code($response) === 200) {
              $body = wp_remote_retrieve_body($response);
              $info_data = json_decode($body, true);
              if (isset($info_data['sections']['changelog'])) {
                  $changelog_content = $info_data['sections']['changelog'];
              }
          }
          echo json_encode(['changelog' => $changelog_content]);
          break;
        default:
          // Default GET request handling
          echo json_encode(get_option($optionsKey));
          break;
      }
      /* $dataArray = get_option($optionsKey); */
      /* unset($dataArray['dataCredentials']); */
      /* echo json_encode($dataArray); */
      break;
    case 'POST':
      $what = isset($_POST['what']) ? $_POST['what'] : null;
      $data = json_decode(stripslashes($_POST['data']), true);

      // echo 'data that will be saved: ' . var_dump($data);
      switch ($what) {
        case 'verify-cache':
          $postId = $data['postId'];
          $postType = get_post_type($postId);
          $permaLink = get_permalink($postId);

          if ($postType === "wp_template") {
            $permaLink = network_home_url('/');
          }
          $path = parse_url($permaLink, PHP_URL_PATH);

          $lastSegment = trim(basename($path));
          $hash = $lastSegment;

          if ($postType === "wp_template") {
            $hash = "bemoto-home-page";
          }

          $currentVehiclesFilePath = dirname(getPathFor('vehicle.html', $hash));
          $result = getLastSyncDateFromUrl($permaLink);
          if ($result->status != 200) {
            $data = [
              'error' => $result->error
            ];

            if ($result->errNo === CURLE_OPERATION_TIMEOUTED || $result->errNo === CURLE_COULDNT_CONNECT) {
              $data['error'] = "It took to long";
            }
            status_header(200); // OK
            wp_send_json($data, 200);
            break;
          }
          $oldLastSyncDate = $result->content;
          $oldLastSyncDateObj = new DateTime($oldLastSyncDate);
          deleteFolder($currentVehiclesFilePath);

          $result = getLastSyncDateFromUrl($permaLink);
          if ($result->status != 200) {
            $data = [
              'error' => $result->error
            ];
            status_header(200); // OK
            wp_send_json($data, 200);
            break;
          }
          $newLastSyncDate = $result->content;
          $newLastSyncDateObj = new DateTime($newLastSyncDate);
          $diffInSeconds = getTimeDiffInSeconds($newLastSyncDateObj, $oldLastSyncDateObj);

          // save result in db
          $bemotoOptionsInDb = get_option($pageCacheKey, []);
          $data = [
            $postId => $diffInSeconds
          ];

          $data = array_replace($bemotoOptionsInDb, $data);
          $updateUnsuccesfull = update_option($pageCacheKey, $data, false);
          if ($updateUnsuccesfull) {
            add_option($pageCacheKey, $data);
          }

          if ($result->status == 200) {
            $data = [
              'diffInSeconds' => $diffInSeconds,
              'oldLastSyncDate' => $oldLastSyncDate,
              'newLastSyncDate' => $newLastSyncDate
            ];
          } else {
            $data = [
              'code' => $result->status,
              'error' => $result->error
            ];
          }
          status_header(200); // OK
          wp_send_json($data, 200);
          break;
        case 'delete-shortcode':
          $postId = $data['postId'];
          delete_shortcode($postId);
          flush_car_rewrite_rules();
          break;
        case 'delete-page':
          $postId = $data['postId'];
          $forceDelete = $data['forceDelete'];
          delete_page($postId, $forceDelete);
          flush_car_rewrite_rules();
          break;
        case 'restore-page':
          $postId = $data['postId'];
          set_page_status($postId, 'draft');
          break;
        case 'new-page':
          $title = $data['title'];
          $shortcode = $data['shortcode'];
          add_bemoto_plugin_page($title, $shortcode);
          flush_car_rewrite_rules();
          break;
        case 'shortcode':
          $postId = $data['postId'];
          $postTitle = $data['postTitle'];
          $shortcode = $data['shortcode'];
          update_shortcode($postId, $postTitle, $shortcode);

          $permaLink = isset($pathSegments[0]) && $data['permaLink'];
          $pathSegments = explode('/', trim($permaLink, '/')); // Split URL into segments
          $basePath = isset($pathSegments[0]) ? '/' . $pathSegments[0] . '/' : '/'; // Get the base path
          $hash = $pathSegments[0];

          $currentVehiclesFilePath = dirname(getPathFor('vehicle.html', $hash));
          echo PHP_EOL . 'deleting folder: ' . $currentVehiclesFilePath . PHP_EOL;
          deleteFolder($currentVehiclesFilePath);
          break;
        default:
          $bemotoOptionsInDb = get_option($optionsKey);

          $result = $data;
          if (is_array($bemotoOptionsInDb)) {
            $result = array_replace($bemotoOptionsInDb, $data);
          }
          update_option($optionsKey, $result, false);
          break;
      }
  }

  wp_die(); // this is required to terminate immediately and return a proper response
}

function post_contains_bemoto_plugin_shortcode($postId)
{
  $post = get_post($postId);
  if (!$post) {
    return false; // Post not found
  }

  $content = $post->post_content;

  // Check if the [bemoto_plugin] shortcode exists in the content
  if (has_shortcode($content, 'bemoto_plugin')) {
    return true;
  }

  return false;
}

add_action(
  'post_updated',
  function ($post_ID, $post_after, $post_before) {
    // Check if the post type is relevant (e.g., 'post', 'page', or a custom post type)
    if (post_contains_bemoto_plugin_shortcode($post_ID)) {
      // echo "The post contains the [bemoto_plugin] shortcode.";
      // Compare the old slug with the new slug
      $old_slug = $post_before->post_name;
      $new_slug = $post_after->post_name;

      if ($old_slug !== $new_slug) {
        $oldFolderName = dirname(getPathFor('vehicle.html', $old_slug));
        $newFolderName = dirname(getPathFor('vehicle.html', $new_slug));
        error_log("Renaming folder: $oldFolderName → $newFolderName");
        rename($oldFolderName, $newFolderName);
      }
    }
  },
  10,
  3
);

// This bit is a special action hook that works with the WordPress AJAX functionality.
add_action('wp_ajax_bemoto_refresh', 'bemoto_refresh');
add_action('wp_ajax_nopriv_bemoto_refresh', 'bemoto_refresh');
add_action('wp_ajax_bemoto_options', 'handle_bemoto_options');
add_action('wp_ajax_bemoto_bootstrap', 'handle_bemoto_bootstrap');

function handle_bemoto_bootstrap() {
  if (!current_user_can('manage_options')) {
    wp_die('Unauthorized');
  }

  global $optionsKey, $backendUrl, $pluginVersion;
  $ownerEmail = $_POST['ownerEmail'] ?? '';
  
  if (empty($ownerEmail) || !is_email($ownerEmail)) {
    wp_send_json_error(['message' => 'Invalid email address.'], 400);
  }

  $options = get_option($optionsKey);
  $mobileDataFetcher = new BackendApi($options, $backendUrl, $pluginVersion);
  
  $newApiKey = $mobileDataFetcher->bootstrap($ownerEmail);

  if ($newApiKey) {
    wp_send_json_success(['apiKey' => $newApiKey]);
  } else {
    wp_send_json_error(['message' => 'Bootstrap failed. Domain might already be registered.'], 403);
  }
}

add_action('template_redirect', 'bemoto_template_redirect_handler');

function bemoto_template_redirect_handler() {
    global $wp_query;

    // Check if it's a 404 and a carId is present in the query vars
    if ($wp_query->is_404 && get_query_var('carId')) {
        // Try to find the page with the shortcode that is also the front page
        $front_page_id = get_option('page_on_front');
        $shortcode_pages = find_all_shortcodes();
        $target_page_id = null;

        foreach ($shortcode_pages as $page) {
            if ($page['postId'] == $front_page_id) {
                $target_page_id = $front_page_id;
                break;
            }
        }

        // If no front page with shortcode, try to find any page with shortcode
        if (is_null($target_page_id) && !empty($shortcode_pages)) {
            $target_page_id = $shortcode_pages[0]['postId']; // Use the first found shortcode page
        }

        // If a target page is found, force WordPress to load it
        if ($target_page_id) {
            $wp_query->is_404 = false;
            $wp_query->is_page = true;
            $wp_query->is_single = true;
            $wp_query->query_vars['page_id'] = $target_page_id;
            $wp_query->queried_object_id = $target_page_id;
            $wp_query->queried_object = get_post($target_page_id);
            status_header(200); // Set HTTP status to 200 OK

            // Set up global post data for the theme
            global $post;
            $post = $wp_query->queried_object;
            setup_postdata($post);

            // Manually apply the_content filter to process shortcodes in the post content
            // This ensures the shortcode output is available when the theme calls the_content()
            
            // Extract car details from query vars
            $carId_from_query = get_query_var('carId');
            $carName_from_query = get_query_var('carName');
            $source_from_query = get_query_var('source');

            // Construct the shortcode with attributes
            $dynamic_shortcode = '[bemoto_plugin';
            if (!empty($carId_from_query)) {
                $dynamic_shortcode .= ' carid="' . esc_attr($carId_from_query) . '" ';
            }
            if (!empty($carName_from_query)) {
                $dynamic_shortcode .= ' carname="' . esc_attr($carName_from_query) . '" ';
            }
            if (!empty($source_from_query)) {
                $dynamic_shortcode .= ' source="' . esc_attr($source_from_query) . '" ';
            }
            $dynamic_shortcode .= ']';

            // Replace the original page content with our dynamic shortcode
            $post->post_content = $dynamic_shortcode;
            $post->post_content = apply_filters('the_content', $post->post_content);
        }
    }
}

// Always register rewrite rules on init
add_action('init', function() {
    global $optionsKey, $backendUrl, $pluginVersion;
    $options = get_option($optionsKey);
    $backendApi = new BackendApi($options, $backendUrl, $pluginVersion);
    add_all_bemoto_rewrite_rules($backendApi);
});

// Flushes rewrite rules after ensuring they are added.
// Used on activation, deactivation, and version update check.
function flush_car_rewrite_rules()
{
  // Rules are already added by the 'init' hook, so we just need to flush.
  flush_rewrite_rules();
}
register_activation_hook(__FILE__, 'flush_rewrite_rules');

// Also flush rules on deactivation to clean up
register_deactivation_hook(__FILE__, 'flush_rewrite_rules');

function bemoto_start_buffering() {
    // Don't buffer on admin, ajax, or cron requests.
    if (is_admin() || wp_doing_ajax() || wp_doing_cron()) {
        return;
    }
    // Start the buffer.
    ob_start();
}

function bemoto_end_buffering_and_set_headers() {
    // Only run if we're on the frontend and a buffer is active.
    if (is_admin() || wp_doing_ajax() || wp_doing_cron() || !ob_get_level()) {
        return;
    }

    global $isProd;

    $content = ob_get_clean();

    // For logged-in users or in debug mode, do nothing.
    if (is_user_logged_in() || !$isProd) {
        echo $content;
        return;
    }

    // For public visitors, check if the shortcode's unique footprint is in the final HTML.
    if (strpos($content, 'data-nonce="') !== false) {
        // Our shortcode ran. Now, get the cache file path.
        $vehicle_filepath = bemoto_get_current_vehicle_filepath();

        if ($vehicle_filepath && file_exists($vehicle_filepath)) {
            // Get the file's modification time.
            $mtime = filemtime($vehicle_filepath);
            $expiry_time = $mtime + 3600; // 1 hour

            // Send our public cache headers.
            if (!headers_sent()) {
                header('Cache-Control: public, max-age=3600');
                header('Expires: ' . gmdate('D, d M Y H:i:s T', $expiry_time));
            }
        }
    }

    echo $content;
}

add_action('init', 'bemoto_start_buffering', 1);
add_action('shutdown', 'bemoto_end_buffering_and_set_headers', 0);
function dbi_bemoto_migration_script() {
    $new_settings = get_option($optionsKey);

    if ( ! empty($new_settings) ) {
        return;
    }

    $old_settings = get_option('dbi_mobile_de_plugin_options');

    if ( ! empty($old_settings) ) {
        update_option('dbi_bemoto_plugin', $old_settings);
    }
}

// Rejestracja funkcji, aby odpaliła się przy włączeniu wtyczki
register_activation_hook(__FILE__, 'dbi_bemoto_migration_script');
