<?php

/**
 * Everything car details related
 **/

// Adds all necessary rewrite rules based on current shortcode usage.
// Called only when rules need to be flushed (activation, update, hash change).
function add_all_bemoto_rewrite_rules($backendApi) { // Added $backendApi parameter
  $pages = find_all_shortcodes(); // Assuming this returns an array like [['postId' => 1, 'permaLink' => '...'], ...]
  if (empty($pages)) {
    return; // No pages, no rules to add
  }

  // Sort pages by the length of their permalink path in descending order.
  // This ensures that more specific rules are matched first.
  usort($pages, function ($a, $b) {
    $path_a = trim(wp_make_link_relative($a['permaLink']), '/');
    $path_b = trim(wp_make_link_relative($b['permaLink']), '/');
    return strlen($path_b) <=> strlen($path_a);
  });

  foreach ($pages as $page) {
    // Get the full relative path, trim leading/trailing slashes for regex construction
    $page_path_for_regex = trim(wp_make_link_relative($page['permaLink']), '/');
    $postId = $page['postId'];

    $regex_prefix = '';
    if (!empty($page_path_for_regex)) {
      $regex_prefix = $page_path_for_regex . '/'; // e.g., 'standorte/albstadt/'
    }

    // The car details slug format is assumed to be 'car-name-CARID-SOURCE'
    // Example: standorte/albstadt/audi-q5-12345-abcdef123456
    // The regex should capture the car name, car ID, and source.
    // (.+?) for car name slug (non-greedy)
    // ([0-9]+) for carId (digits)
    // ([a-f0-9]+) for source (hex characters)
    $rule_regex = '^' . $regex_prefix . '(.+?)-([0-9]+)-([a-f0-9]+)/?$';

    // If it's the home page (empty prefix), add a more specific rule with 'top' priority
    if (empty($page_path_for_regex)) {
        add_rewrite_rule('^(.+?)-([0-9]+)-([a-f0-9]+)/?$', 'index.php?page_id=' . $postId . '&carName=$matches[1]&carId=$matches[2]&source=$matches[3]', 'top');
        $backendApi->sendDebugLog(
            'Bemoto Rewrite Rule Added (HOME PAGE TOP PRIORITY): Regex: ^(.+?)-([0-9]+)-([a-f0-9]+)/?$, Target: index.php?page_id=' . $postId . '&carName=$matches[1]&carId=$matches[2]&source=$matches[3]'
        );
    }

    // Define the target query string
    // $matches[1] will be the car name slug
    // $matches[2] will be carId
    // $matches[3] will be source
    $rule_target = 'index.php?page_id=' . $postId . '&carName=$matches[1]&carId=$matches[2]&source=$matches[3]';

    add_rewrite_rule($rule_regex, $rule_target, 'top');
    $backendApi->sendDebugLog(
      'Bemoto Rewrite Rule Added: Page Path: ' . $page_path_for_regex . ', Regex: ' . $rule_regex . ', Target: ' . $rule_target
    );

  }

  // Add a generic fallback rule for car details, pointing to the main WordPress index.
  // This rule should be less specific than the ones generated for actual pages.
  // It matches the car details slug at the end of any path.
  $generic_car_details_regex = '^(?:.+?/)?(.+?)-([0-9]+)-([a-f0-9]+)/?';
  $generic_car_details_target = 'index.php?carName=$matches[1]&carId=$matches[2]&source=$matches[3]';

  // Add this rule without 'top' priority to ensure it's evaluated after other rules.
  add_rewrite_rule($generic_car_details_regex, $generic_car_details_target);
  $backendApi->sendDebugLog(
      'Bemoto Generic Car Details Rewrite Rule Added (FALLBACK PRIORITY): Regex: ' . $generic_car_details_regex . ', Target: ' . $generic_car_details_target
  );
}


// Checks if rules need updating based on hash and flushes if necessary.
// Hooked to admin_init for performance.
function check_and_flush_bemoto_rules_on_admin()
{
  // Only run the check for users who can edit posts/pages
  if (!current_user_can('edit_posts')) {
    return; // Exit early for users who cannot edit content (e.g., Subscribers, logged-out users)
  }

  // Option name to store the hash
  $hash_option_name = 'bemoto_rewrite_rules_hash';

  // Get pages with the shortcode
  $pages = find_all_shortcodes(); // Assuming this returns an array like [['postId' => 1, 'permaLink' => '...'], ...]

  // Prepare data for hashing - sort to ensure consistency
  $data_to_hash = [];
  if (!empty($pages)) {
    // Sort pages by postId to ensure consistent order for hashing
    usort($pages, function ($a, $b) {
      return $a['postId'] <=> $b['postId'];
    });
    foreach ($pages as $page) {
      // Use the full relative path for hashing
      $data_to_hash[] = $page['postId'] . ':' . trim(wp_make_link_relative($page['permaLink']), '/');
    }
  }

  // Make sure the plugin version is included in the hash to force a flush on update
  global $optionsKey, $backendUrl, $pluginVersion;
  $current_hash = md5(implode('|', $data_to_hash) . '|' . $pluginVersion);

  // Get the previously stored hash
  $stored_hash = get_option($hash_option_name, '');
  $options = get_option($optionsKey);
  $backendApi = new BackendApi($options, $backendUrl, $pluginVersion);

  // Compare hashes
  if ($current_hash !== $stored_hash) {
    $backendApi->sendDebugLog(
      'Bemoto Rewrite Rules: Hash mismatch detected. Current: ' . $current_hash . ', Stored: ' . $stored_hash
    );

    // Step 1: Add the rules based on the current state
    add_all_bemoto_rewrite_rules($backendApi); // Pass the instance

    // Step 2: Flush the rewrite rules - expensive operation!
    flush_rewrite_rules();
    $backendApi->sendDebugLog(
      'Bemoto Rewrite Rules: Flushing rewrite rules.'
    );

    // Step 3: Store the new hash
    update_option($hash_option_name, $current_hash);
  } else {
    $backendApi->sendDebugLog(
      'Bemoto Rewrite Rules: Hashes match. No flush needed. Current: ' . $current_hash
    );
  }
}
// Hook the check to admin_init instead of init
add_action('admin_init', 'check_and_flush_bemoto_rules_on_admin');


function add_car_query_var($vars)
{
  // Add query variables needed for the car details page
  $vars[] = 'carName'; // Added for the new rule structure
  $vars[] = 'carId';
  $vars[] = 'source';
  return $vars;
}
add_filter('query_vars', 'add_car_query_var');

function get_vehicle_details($locale, $carId, $shortCodeParams='', $source = '', $forceRefresh = false)
{
  $source = preg_replace('/[^a-fA-F0-9]/', '', $source); // Sanitize source to allow only hexadecimal characters
  global $optionsKey, $backendUrl, $pluginVersion, $isProd;
  $options = get_option($optionsKey);
  $mobileDataFetcher = new BackendApi($options, $backendUrl, $pluginVersion);

  $currentVehiclesFilePathHtml = getPathFor('vehicle.html', 'details/'.$carId, $locale);
  
  if ($isProd && !$forceRefresh && file_exists($currentVehiclesFilePathHtml)) {
    $mtime = filemtime($currentVehiclesFilePathHtml);
    if ((time() - $mtime) < 3600) { // 1 hour cache in production
        return;
    }
  }

  $apiEndpoint = '/api/cars/'.$carId.'/'.$source;
  $result = $mobileDataFetcher->getData($locale, $shortCodeParams, $apiEndpoint);
  log_info_in_browser('got results from bemoto server');

  if ($result->errNo === CURLE_OPERATION_TIMEOUTED || $result->errNo === CURLE_COULDNT_CONNECT) {
    // try once more after first timeout
    $result = $mobileDataFetcher->getData($locale, $shortCodeParams, $apiEndpoint);

    // if timed out second time then we don't try anymore
    if ($result->errNo === CURLE_OPERATION_TIMEOUTED || $result->errNo === CURLE_COULDNT_CONNECT) {
      log_error("Couldn't connect to server, timeout happened. Error code: " . $result->errNo . ", message: " . $result->error);
      return;
    }
  }

  //process result if there was no timeouts
  switch ($result->status) {
    case 401:
      log_error("Unauthorized request to backend for car details. Please check your settings.");
      break;
    case 200:
      // Ensure the directory exists before writing the file
      $dirPath = dirname($currentVehiclesFilePathHtml);
      if (!file_exists($dirPath)) {
        wp_mkdir_p($dirPath);
      }
      $fileSaveResult = @file_put_contents($currentVehiclesFilePathHtml, $result->content);
      if ($fileSaveResult === false) {
        $error = error_get_last();
        $errorMessage =  $error['message'];
        log_error_in_browser($errorMessage);
        log_error('Error writing to file: ' . $error['message']);
      }
      break;
    default:
      $errorMessage = $result->content;
      log_error_in_browser($errorMessage);
      if ($result->status == 0) {
        $errorMessage = "Can't connect to firebase";
      } else {
        log_error("Error status: " . $result->status);
      }
      log_error($errorMessage);
  }
}
