<?php

function find_all_shortcodes($showTrashedPages = false)
{
  global $wpdb;
  $shortcode_tag = 'bemoto_plugin'; // the name of the shortcode to search for

  // Query the database for all posts containing the shortcode
  $results = $wpdb->get_results(
    $wpdb->prepare(
      "SELECT ID, post_title, post_content, post_status, post_name, post_type
      FROM $wpdb->posts 
      WHERE post_content LIKE %s
      AND post_status != 'inherit' AND post_status != 'trash'
      /* GROUP BY ID -- Removed as it's not needed for this part of the query and could be problematic without joining meta table. 
         The main purpose here is to get posts with the shortcode in post_content.
         Elementor data is handled separately by WP_Query. */
      ORDER BY post_title ASC",
      '%' . $wpdb->esc_like('[' . $shortcode_tag) . '%'
    )
  );

  $found_shortcodes = [];
  $processed_post_ids = []; // To avoid processing a post multiple times if found by both methods

  // Method 1: Check post_content (already queried)
  foreach ($results as $post) {
    if (has_shortcode($post->post_content, $shortcode_tag)) {
      preg_match_all('/\[' . $shortcode_tag . '[^\]]*\]/', $post->post_content, $matches);
      foreach ($matches[0] as $shortcode_instance) {
        $url = get_permalink($post->ID);
        if ($post->post_type == "wp_template") {
          global $wp_rewrite;
          if (isset($wp_rewrite) && is_object($wp_rewrite) && $wp_rewrite->using_permalinks()) {
            $url = untrailingslashit(network_home_url('/'));
          } else {
            // Fallback if permalinks are not yet initialized, use home URL directly
            $url = home_url('/');
          }
        }
        $found_shortcodes[] = [
          'postId' => $post->ID,
          'postTitle' => $post->post_title,
          'postName' => $post->post_name,
          'permaLink' => $url,
          'shortcode' => $shortcode_instance,
          'status' => $post->post_status
        ];
        $processed_post_ids[$post->ID] = true;
        break; // Found in post_content, no need to check Elementor data for this shortcode instance
      }
    }
  }

  // Method 2: Check Elementor data for posts not yet processed or not found via post_content
  $elementor_posts_query = new WP_Query([
    'post_type' => 'page', // Or any relevant post types
    'post_status' => 'publish,draft,private', // Include relevant statuses
    'posts_per_page' => -1,
    'meta_query' => [
      [
        'key' => '_elementor_data',
        'compare' => 'EXISTS',
      ],
    ],
    'fields' => 'ids', // Only get IDs for performance
  ]);

  if ($elementor_posts_query->have_posts()) {
    foreach ($elementor_posts_query->posts as $post_id) {
      if (isset($processed_post_ids[$post_id])) {
        continue; // Already processed via post_content
      }

      $elementor_data_json = get_post_meta($post_id, '_elementor_data', true);
      if ($elementor_data_json) {
        $elementor_data_array = json_decode($elementor_data_json, true);
        if (json_last_error() === JSON_ERROR_NONE && is_array($elementor_data_array)) {
          if (bemoto_find_shortcode_in_elementor_recursive($elementor_data_array, $shortcode_tag)) {
            $post_obj = get_post($post_id); // Get post object for details
            // Assuming one shortcode instance per page for simplicity in Elementor context for rewrite rule generation
            $shortcode_instance_placeholder = '[' . $shortcode_tag . ']'; // Placeholder, actual attributes might not be easily extractable here
             $url = get_permalink($post_id);
             if ($post_obj->post_type == "wp_template") {
                $url = untrailingslashit(network_home_url('/'));
             }
            $found_shortcodes[] = [
              'postId' => $post_id,
              'postTitle' => $post_obj->post_title,
              'postName' => $post_obj->post_name,
              'permaLink' => $url,
              'shortcode' => $shortcode_instance_placeholder,
              'status' => $post_obj->post_status
            ];
            $processed_post_ids[$post_id] = true;
          }
        }
      }
    }
  }
  wp_reset_postdata();
  
  // Deduplicate based on postId in case a page has shortcode in both post_content and Elementor data
  $final_shortcodes = [];
  $final_post_ids = [];
  foreach ($found_shortcodes as $sc) {
      if (!in_array($sc['postId'], $final_post_ids)) {
          $final_shortcodes[] = $sc;
          $final_post_ids[] = $sc['postId'];
      }
  }

  return $final_shortcodes;
}

// Helper function to recursively find a shortcode tag in Elementor data
function bemoto_find_shortcode_in_elementor_recursive($elements_array, $shortcode_tag_to_find) {
    foreach ($elements_array as $element) {
        if (!empty($element['widgetType'])) {
            // Check Elementor's native Shortcode widget
            if ($element['widgetType'] === 'shortcode' && !empty($element['settings']['shortcode']) && strpos($element['settings']['shortcode'], '[' . $shortcode_tag_to_find) !== false) {
                return true;
            }
            // Check Text Editor widget (common place for shortcodes)
            if ($element['widgetType'] === 'text-editor' && !empty($element['settings']['editor']) && strpos($element['settings']['editor'], '[' . $shortcode_tag_to_find) !== false) {
                return true;
            }
            // Add checks for other common widgets if necessary (e.g., HTML widget)
            if ($element['widgetType'] === 'html' && !empty($element['settings']['html']) && strpos($element['settings']['html'], '[' . $shortcode_tag_to_find) !== false) {
                return true;
            }
        }

        // Generic check in all settings of an element (less performant, more thorough)
        if (!empty($element['settings'])) {
            foreach ($element['settings'] as $setting_value) {
                if (is_string($setting_value) && strpos($setting_value, '[' . $shortcode_tag_to_find) !== false) {
                    return true; // Found the shortcode string in a setting
                }
            }
        }

        // Recursively check inner elements
        if (!empty($element['elements'])) {
            if (bemoto_find_shortcode_in_elementor_recursive($element['elements'], $shortcode_tag_to_find)) {
                return true;
            }
        }
    }
    return false;
}

function update_shortcode($postId, $postTitle, $shortcode)
{
  $post = get_post($postId);

  if (!$post) {
    echo "Post not found.";
    return;
  }

  $content = $post->post_content;

  // Update the shortcode in standard content
  $updated_content = preg_replace('/\[bemoto_plugin[^\]]*\]/', $shortcode, $content);

  // Update the post title and content
  wp_update_post(array(
    'ID'           => $postId,
    'post_content' => $updated_content,
    'post_title'   => $postTitle,
  ), true);

  // Check if Elementor is active on this post
  $elementor_data = get_post_meta($postId, '_elementor_data', true);

  if ($elementor_data) {
    // Decode Elementor data
    $elementor_data_array = json_decode($elementor_data, true);

    if (json_last_error() === JSON_ERROR_NONE && is_array($elementor_data_array)) { // Added is_array check
      // Recursively update shortcodes in Elementor data
      $elementor_data_array = update_elementor_shortcode($elementor_data_array, $shortcode);

      // Save the updated Elementor data
      update_post_meta($postId, '_elementor_data', wp_slash(json_encode($elementor_data_array)));

      // Force Elementor to recognize the post as modified, if Elementor is active
      if (class_exists('\Elementor\Plugin') && \Elementor\Plugin::instance()) {
        $elementor_plugin_instance = \Elementor\Plugin::instance();
        
        update_post_meta($postId, '_elementor_edit_mode', 'builder');
        
        // Safely get and set Elementor version
        $elementor_version_to_set = '';
        if (method_exists($elementor_plugin_instance, 'get_version')) {
            $elementor_version_to_set = $elementor_plugin_instance->get_version();
        } elseif (defined('ELEMENTOR_VERSION')) { // Fallback, though get_version() is preferred
            $elementor_version_to_set = ELEMENTOR_VERSION;
        }
        if (!empty($elementor_version_to_set)) {
            update_post_meta($postId, '_elementor_version', $elementor_version_to_set);
        }

        update_post_meta($postId, '_elementor_template_type', 'wp-post'); // Or appropriate type
        update_post_meta($postId, '_elementor_css', ''); // Clear existing CSS

        // 🔥 **Clear Elementor Cache (Fixes Caching Issue!)**
        if (isset($elementor_plugin_instance->files_manager) && method_exists($elementor_plugin_instance->files_manager, 'clear_cache')) {
          $elementor_plugin_instance->files_manager->clear_cache();
        }

        // 🔥 **Clear Elementor Element Cache (CRUCIAL FIX)**
        // This transient might be specific to Elementor Pro or certain caching setups.
        delete_transient('elementor_pro_page_block_cache_' . $postId); 
        
        // Additional Elementor cache clearing if available
        // if (isset($elementor_plugin_instance->posts_css_manager) && method_exists($elementor_plugin_instance->posts_css_manager, 'clear_cache')) {
        //   $elementor_plugin_instance->posts_css_manager->clear_cache();
        // }
      }
    }
  }
  // Clear WordPress cache for the post
  clean_post_cache($postId);
  wp_cache_flush();
}

function update_elementor_shortcode($data, $shortcode)
{
  foreach ($data as &$element) {
    if (isset($element['settings']) && is_array($element['settings'])) {
      foreach ($element['settings'] as &$setting) {
        if (is_string($setting)) {
          $setting = preg_replace('/\[bemoto_plugin[^\]]*\]/', $shortcode, $setting);
        }
      }
    }

    // Check for nested elements
    if (isset($element['elements']) && is_array($element['elements'])) {
      $element['elements'] = update_elementor_shortcode($element['elements'], $shortcode);
    }
  }

  return $data;
}

function parseShortcode($input)
{
  // Extract the attributes part of the string (remove the shortcode tag name)
  if (preg_match('/\[bemoto_plugin\s+(.*)\]/', $input, $matches)) {
    $attributeString = $matches[1];
  } else {
    throw new Exception("Invalid input format");
  }

  // Regex to match attribute key-value pairs
  preg_match_all('/(\w+)="([^"]*)"/', $attributeString, $matches, PREG_SET_ORDER);

  $result = [];

  // Loop through matches and store in result array with lowercase keys
  foreach ($matches as $match) {
    $key = strtolower($match[1]); // Convert key to lowercase
    $value = $match[2];           // Extract the value
    $result[$key] = $value;
  }

  return $result;
}

function delete_shortcode($postId)
{
  $post = get_post($postId);
  $content = $post->post_content;

  // Update the shortcode in the content
  $updated_content = preg_replace('/\[bemoto_plugin[^\]]*\]/', '', $content);

  // Update the post with the new content
  wp_update_post(array(
    'ID' => $postId,
    'post_content' => $updated_content
  ), true);

  if (is_wp_error($postId)) {
    $errors = $postId->get_error_messages();
    foreach ($errors as $error) {
      echo $error;
    }
  }
}

function delete_page($postId, $forceDelete = false)
{
  // Delete the page
  wp_delete_post($postId, $forceDelete); // The second argument true means "force delete", bypassing the trash

  // Check if the deletion was successful
  if (get_post($postId) === null) {
    echo 'Page deleted successfully!';
  } else {
    echo 'Failed to delete page.';
  }
}

function set_page_status($postId, $postStatus)
{
  wp_update_post(array(
    'ID' => $postId,
    'post_status' => $postStatus,
  ), true);

  if (is_wp_error($postId)) {
    $errors = $postId->get_error_messages();
    foreach ($errors as $error) {
      echo $error;
    }
  }
}
